/* $Id: heaptuple.c,v 1.5 2005/05/12 19:29:35 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/tup.c,v 1.3 2004/10/12 22:04:13 flaw
 * if/src/tup.c,v 1.19 2004/09/28 15:41:51 flaw
 *//*
 * Postgres HeapTuple interface for Python
 *
 * This implements an interface to HeapTuples that are of a TupleDesc that does
 * not specify the columns of a Relation.
 *
 * It is the interface for HeapTuples that are not a registered composite type.
 */
#include <setjmp.h>
#include <postgres.h>
#include <tcop/tcopprot.h>
#include <access/htup.h>
#include <access/hio.h>
#include <access/heapam.h>
#include <access/tupdesc.h>
#include <catalog/pg_type.h>
#include <lib/stringinfo.h>
#include <parser/parse_type.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/utils.h>
#include <pypg/conv.h>
#include <pypg/tif.h>

static int
tup_length(PyObj self)
{
	return(PyPgHeapTuple_FetchNatts(self));
}

static PyObj
tup_item(PyObj self, int attnum)
{
	TupleDesc td;
	HeapTuple ht;
	PyObj rob;

	td = PyPgHeapTuple_FetchTupleDesc(self);
	ht = PyPgHeapTuple_FetchHeapTuple(self);
	rob = PyPgTIF_FetchItem(td, ht, (unsigned int) attnum);

	return(rob);
}

static PyObj
tup_slice(PyObj self, int from, int to)
{
	TupleDesc td;
	HeapTuple ht;
	PyObj rob;

	td = PyPgHeapTuple_FetchTupleDesc(self);
	ht = PyPgHeapTuple_FetchHeapTuple(self);
	rob = PyPgTIF_FetchSlice(td, ht, (unsigned int) from, (unsigned int) to);

	return(rob);
}

static PySequenceMethods PyPgHeapTupleAsSequence = {
	tup_length,			/* sq_length */
	NULL,					/* sq_concat */
	NULL,					/* sq_repeat */
	tup_item,			/* sq_item */
	tup_slice,			/* sq_slice */
	NULL,					/* sq_ass_item */
	NULL,					/* sq_ass_slice */
	NULL,					/* sq_contains */
	NULL,					/* sq_inplace_concat */
	NULL,					/* sq_inplace_repeat */
};

static PyObj
tup_subscript(PyObj self, PyObj sub)
{
	TupleDesc td;
	HeapTuple ht;
	PyObj rob;

	td = PyPgHeapTuple_FetchTupleDesc(self);
	ht = PyPgHeapTuple_FetchHeapTuple(self);
	rob = PyPgTIF_FetchSubscript(td, ht, sub);

	return(rob);
}

static PyMappingMethods PyPgHeapTupleAsMapping = {
	tup_length,				/* mp_length */
	tup_subscript,			/* mp_subscript */
	NULL,						/* mp_ass_subscript */
};

static PyMemberDef PyPgHeapTuple_Members[] = {
	{"TupleDesc", T_OBJECT, offsetof(struct PyPgHeapTuple, ht_desc), RO,
	"the HeapTuple's TupleDesc"},

	{"oid", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's oid"},
	{"tableoid", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's tableoid"},

	{"Xmin", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Xmin"},
	{"Xmax", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Xmax"},
	{"Xvac", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Xvac"},

	{"Cmin", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Cmin"},
	{"Cmax", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's Cmax"},

	{"TID", T_OBJECT, offsetof(struct PyPgHeapTuple, ob_type), RO,
	"the HeapTuple's TID"},
	{NULL}
};

static int
tup_traverse(PyObj self, visitproc visit, void *arg)
{
	int err = 0;
	err = visit(PyPgHeapTuple_FetchPyPgTupleDesc(self), arg);
	return(err);
}

static void
tup_dealloc(PyObj self)
{
	HeapTuple ht;
	PyObj ob;

	ht = PyPgHeapTuple_FetchHeapTuple(self);
	if (ht != NULL)
	{
		heap_freetuple(ht);
		PyPgHeapTuple_FixHeapTuple(self, NULL);
	}

	ob = PyPgHeapTuple_FetchPyPgTupleDesc(self);
	if (ob != NULL)
	{
		DECREF(ob);
		PyPgHeapTuple_FixPyPgTupleDesc(self, NULL);
	}

	self->ob_type->tp_free(self);
}

static PyObj
tup_getattro(PyObj self, PyObj attro)
{
	PyObj rob = NULL;
	char *attr = PyString_AS_STRING(attro);

	if (!strcmp(attr, "oid"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(OIDOID,
					PyPgHeapTuple_FetchOid(self));
	}
	else if (!strcmp(attr, "tableoid"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(OIDOID,
					PyPgHeapTuple_FetchTableOid(self));
	}
	else if (!strcmp(attr, "Xmin"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(XIDOID,
					PyPgHeapTuple_FetchXmin(self));
	}
	else if (!strcmp(attr, "Xmax"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(XIDOID,
					PyPgHeapTuple_FetchXmax(self));
	}
	else if (!strcmp(attr, "Xvac"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(XIDOID,
					PyPgHeapTuple_FetchXvac(self));
	}
	else if (!strcmp(attr, "Cmin"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(CIDOID,
					PyPgHeapTuple_FetchCmin(self));
	}
	else if (!strcmp(attr, "Cmax"))
	{
		rob = PyPgObject_FromTypeOidAndDatum(CIDOID,
					PyPgHeapTuple_FetchCmax(self));
	}
	else if (!strcmp(attr, "TID"))
	{
		ItemPointer to, from;
		from = &PyPgHeapTuple_FetchTID(self);
		to = palloc(SizeOfIptrData);
		if (to)
		{
			ItemPointerCopy(from, to);
			rob = PyPgObject_FromTypeOidAndDatum(TIDOID, PointerGetDatum(to));
		}
	}
	else
		rob = PyObject_GenericGetAttr(self, attro);

	return(rob);
}

static int
tup_compare(PyObj self, PyObj that)
{
	unsigned long i;
	TupleDesc selftupd, thattupd;

	selftupd = PyPgHeapTuple_FetchTupleDesc(self);
	thattupd = PyPgHeapTuple_FetchTupleDesc(that);

	if (selftupd->natts != thattupd->natts)
		return(-1);

	for (i = 0; i < selftupd->natts; ++i)
		if ((selftupd->attrs[i])->atttypid != (thattupd->attrs[i])->atttypid)
			return(-1);

	return(-1);
}

static PyObj
tup_repr(PyObj self)
{
	StringInfoData buf;
	unsigned long i, items;
	PyObj rob;

	items = PyPgHeapTuple_FetchNatts(self);
	initStringInfo(&buf);
	appendStringInfoChar(&buf, '(');

	for (i = 0; i < items; ++i)
	{
		char *str = NULL;
		PyObj ci, obstr;
		ci = self->ob_type->tp_as_sequence->sq_item(self, i);
		obstr = PyObject_Repr(ci);
		str = PyString_AS_STRING(obstr);
		appendStringInfoString(&buf, str);

		DECREF(obstr);
		DECREF(ci);

		if (i+1 < items)
			appendStringInfoString(&buf, ", ");
	}

	appendStringInfoChar(&buf, ')');
	rob = PyString_FromStringAndSize(buf.data, buf.len);
	pfree(buf.data);
	return(rob);
}

static PyObj
tup_str(PyObj self)
{
	HeapTuple ht;
	TupleDesc td;
	char *str;
	PyObj rob;
	
	td = PyPgHeapTuple_FetchTupleDesc(self);
	ht = PyPgHeapTuple_FetchHeapTuple(self);
	str = PyPgTIF_String(td, ht);
	rob = PyString_FromString(str);
	pfree(str);

	return(rob);
}

const char PyPgHeapTuple_Doc[] = "Postgres HeapTuple interface type";
PyTypeObject PyPgHeapTuple_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.HeapTuple",			/* tp_name */
	sizeof(struct PyPgHeapTuple),	/* tp_basicsize */
	0,										/* tp_itemsize */
	tup_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	tup_compare,						/* tp_compare */
	tup_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	&PyPgHeapTupleAsSequence,		/* tp_as_sequence */
	&PyPgHeapTupleAsMapping,		/* tp_as_mapping */
	NULL,									/* tp_hash */
	NULL,									/* tp_call */
	tup_str,								/* tp_str */
	tup_getattro,						/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) PyPgHeapTuple_Doc,	/* tp_doc */
	tup_traverse,						/* tp_traverse */
	(inquiry) NULL,					/* tp_clear */
	(richcmpfunc) NULL,				/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	(getiterfunc) PySeqIter_New,	/* tp_iter */
	NULL,									/* tp_iternext */
	NULL,									/* tp_methods */
	PyPgHeapTuple_Members,			/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	NULL,									/* tp_new */
};

PyObj
PyPgHeapTuple_Initialize(PyObj self, PyObj tupd, HeapTuple tup)
{
	if (self == NULL)
		return(NULL);

	PyPgHeapTuple_FixHeapTuple(self, tup);
	PyPgHeapTuple_FixPyPgTupleDesc(self, tupd);
	INCREF(tupd);

	return(self);
}
/*
 * vim: ts=3:sw=3:noet:
 */
